<?php
/**
 * Plugin Name: Variably Pretty
 * Plugin URI: https://bloodweb.net
 * Description: Renders WooCommerce variable product options as interactive pills instead of dropdowns
 * Version: 1.0.4
 * Author: Jack Ewers
 * Author URI: https://bloodweb.net
 * License: GPL v2 or later
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * WC requires at least: 4.0
 * WC tested up to: 8.0
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Check if WooCommerce is active
if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
    add_action('admin_notices', function() {
        echo '<div class="error"><p><strong>Variably Pretty</strong> requires WooCommerce to be installed and active.</p></div>';
    });
    return;
}

class Variably_Pretty {
    
    private static $instance = null;
    private static $assets_printed = false;
    
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Admin settings
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        
        // Frontend hooks
        add_action('wp_footer', [$this, 'print_inline_assets']);
        add_filter('woocommerce_dropdown_variation_attribute_options_html', [$this, 'render_pill_variations'], 10, 2);
    }
    
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            'Variably Pretty Settings',
            'Variably Pretty',
            'manage_woocommerce',
            'variably-pretty',
            [$this, 'settings_page']
        );
    }
    
    public function register_settings() {
        register_setting('variably_pretty_options', 'variably_pretty_enabled_products', [
            'type' => 'array',
            'default' => []
        ]);
        
        register_setting('variably_pretty_options', 'variably_pretty_enable_all', [
            'type' => 'boolean',
            'default' => false
        ]);
    }
    
    public function settings_page() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }
        
        // Save settings
        if (isset($_POST['variably_pretty_save']) && check_admin_referer('variably_pretty_settings')) {
            $enable_all = isset($_POST['variably_pretty_enable_all']) ? 1 : 0;
            $enabled_products = isset($_POST['variably_pretty_enabled_products']) ? array_map('intval', $_POST['variably_pretty_enabled_products']) : [];
            
            update_option('variably_pretty_enable_all', $enable_all);
            update_option('variably_pretty_enabled_products', $enabled_products);
            
            echo '<div class="updated"><p>Settings saved successfully!</p></div>';
        }
        
        $enable_all = get_option('variably_pretty_enable_all', false);
        $enabled_products = get_option('variably_pretty_enabled_products', []);
        
        // Get all variable products
        $variable_products = wc_get_products([
            'type' => 'variable',
            'limit' => -1,
            'orderby' => 'title',
            'order' => 'ASC'
        ]);
        
        ?>
        <div class="wrap">
            <h1>Variably Pretty Settings</h1>
            <p>Configure which variable products should display options as interactive pills instead of dropdowns.</p>
            
            <form method="post" action="">
                <?php wp_nonce_field('variably_pretty_settings'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="variably_pretty_enable_all">Enable for All Products</label>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox" 
                                       id="variably_pretty_enable_all" 
                                       name="variably_pretty_enable_all" 
                                       value="1" 
                                       <?php checked($enable_all, 1); ?> />
                                Apply pill-style variations to all variable products
                            </label>
                            <p class="description">When enabled, all variable products will use pill-style selection.</p>
                        </td>
                    </tr>
                    
                    <tr id="product-selector-row">
                        <th scope="row">
                            <label>Select Products</label>
                        </th>
                        <td>
                            <p class="description" style="margin-bottom: 10px;">Choose specific products to enable pill-style variations:</p>
                            
                            <div style="max-height: 400px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; background: #f9f9f9;">
                                <?php if (empty($variable_products)): ?>
                                    <p>No variable products found. <a href="<?php echo admin_url('post-new.php?post_type=product'); ?>">Create one</a>.</p>
                                <?php else: ?>
                                    <?php foreach ($variable_products as $product): ?>
                                        <label style="display: block; padding: 5px 0;">
                                            <input type="checkbox" 
                                                   name="variably_pretty_enabled_products[]" 
                                                   value="<?php echo esc_attr($product->get_id()); ?>"
                                                   <?php checked(in_array($product->get_id(), $enabled_products)); ?> />
                                            <?php echo esc_html($product->get_name()); ?>
                                            <span style="color: #666; font-size: 0.9em;">(ID: <?php echo $product->get_id(); ?>)</span>
                                        </label>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                </table>
                
                <p class="submit">
                    <input type="submit" name="variably_pretty_save" class="button-primary" value="Save Settings" />
                </p>
            </form>
            
            <script>
            document.addEventListener('DOMContentLoaded', function() {
                const enableAllCheckbox = document.getElementById('variably_pretty_enable_all');
                const productRow = document.getElementById('product-selector-row');
                
                function toggleProductSelector() {
                    if (enableAllCheckbox.checked) {
                        productRow.style.opacity = '0.5';
                        productRow.style.pointerEvents = 'none';
                    } else {
                        productRow.style.opacity = '1';
                        productRow.style.pointerEvents = 'auto';
                    }
                }
                
                enableAllCheckbox.addEventListener('change', toggleProductSelector);
                toggleProductSelector();
            });
            </script>
        </div>
        <?php
    }
    
    public function print_inline_assets() {
        if (!is_product()) {
            return;
        }
        
        global $product;
        if (!$product || !$product->is_type('variable')) {
            return;
        }
        
        if (!$this->is_enabled_for_product($product->get_id())) {
            return;
        }
        
        if (self::$assets_printed) {
            return;
        }
        
        self::$assets_printed = true;
        
        ?>
        <style id="variably-pretty-styles">
        .variably-pretty-wrapper{margin-bottom:25px}.variably-pretty-label{font-weight:600;font-size:16px;margin-bottom:12px;color:#333;text-transform:capitalize}.variably-pretty-pills{display:flex;flex-wrap:wrap;gap:10px;margin-top:15px;margin-bottom:8px}.variably-pretty-pill{padding:10px 20px;border:2px solid #ddd;border-radius:10px;background:#fff;color:#333;font-size:14px;font-weight:500;cursor:pointer;transition:all .2s ease;outline:0;white-space:nowrap}.variably-pretty-pill:focus{outline:0}.variably-pretty-pill:hover:not(.disabled):not(:focus){border-color:#666;transform:translateY(-2px);box-shadow:0 2px 8px rgba(0,0,0,.1)}.variably-pretty-pill.selected{background:#000;color:#fff;border-color:#000}.variably-pretty-pill.selected:hover{background:#333;border-color:#333}.variably-pretty-pill.disabled{opacity:.3;cursor:not-allowed;pointer-events:none;background:#f5f5f5;color:#999;border-color:#e0e0e0}@media (max-width:768px){.variably-pretty-pill{padding:8px 16px;font-size:13px}.variably-pretty-pills{gap:8px}}.variably-pretty-pill{animation:fadeIn .3s ease}@keyframes fadeIn{from{opacity:0;transform:scale(.95)}to{opacity:1;transform:scale(1)}}.variably-pretty-wrapper.theme-light .variably-pretty-pill{border-color:#e5e5e5}.variably-pretty-wrapper.theme-dark .variably-pretty-pill{background:#1a1a1a;color:#fff;border-color:#333}.variably-pretty-wrapper.theme-dark .variably-pretty-pill:hover:not(.disabled){border-color:#666}.variations select.variably-pretty-hidden-select{display:none!important}.variably-pretty-wrapper::after{content:'';display:block;width:100%;height:1px;background:#e0e0e0;margin-top:20px}.variably-pretty-wrapper:last-child::after{display:none}
        </style>
        <script id="variably-pretty-script">
        (function($){'use strict';class VariablyPretty{constructor(){this.form=$('form.variations_form');if(!this.form.length)return;const self=this;setTimeout(()=>{self.init()},100)}init(){this.setupPillClicks();this.setupVariationListeners();$('.variably-pretty-hidden-select').each(function(){$(this).addClass('variations-select-hidden')});this.updateAvailability()}setupPillClicks(){const self=this;$(document.body).on('click','.variably-pretty-pill',function(e){e.preventDefault();e.stopPropagation();const $pill=$(this);console.log('Pill clicked:',$pill.text(),'Value:',$pill.data('value'));if($pill.hasClass('disabled')){console.log('Pill is disabled, ignoring');return}const $wrapper=$pill.closest('.variably-pretty-wrapper');const $hiddenSelect=$wrapper.find('.variably-pretty-hidden-select');const value=$pill.data('value');console.log('Hidden select found:',$hiddenSelect.length,'Current value:',$hiddenSelect.val());if($pill.hasClass('selected')){console.log('Deselecting pill');$pill.removeClass('selected');$hiddenSelect.val('').trigger('change').trigger('focusin').trigger('woocommerce_variation_select_change')}else{console.log('Selecting pill with value:',value);$wrapper.find('.variably-pretty-pill').removeClass('selected');$pill.addClass('selected');$hiddenSelect.val(value).trigger('change').trigger('focusin').trigger('woocommerce_variation_select_change')}console.log('New select value:',$hiddenSelect.val());self.form.trigger('check_variations');self.form.trigger('woocommerce_variation_has_changed');setTimeout(()=>self.updateAvailability(),100)})}setupVariationListeners(){const self=this;this.form.on('woocommerce_update_variation_values',function(){self.updateAvailability()});this.form.on('found_variation',function(event,variation){console.log('Variation found:',variation)});this.form.on('reset_data',function(){$('.variably-pretty-pill').removeClass('selected disabled');self.updateAvailability()})}updateAvailability(){const self=this;const selections={};$('.variably-pretty-hidden-select').each(function(){const $select=$(this);const attrName=$select.attr('name');const value=$select.val();if(value){selections[attrName]=value}});const variationData=this.form.data('product_variations');if(!variationData||variationData.length===0){const variations=window.product_variations||[];this.updatePillAvailability(selections,variations)}else{this.updatePillAvailability(selections,variationData)}}updatePillAvailability(selections,variations){$('.variably-pretty-wrapper').each(function(){const $wrapper=$(this);const attributeName=$wrapper.data('attribute');$wrapper.find('.variably-pretty-pill').each(function(){const $pill=$(this);const optionValue=$pill.data('value');const isAvailable=isOptionAvailable(attributeName,optionValue,selections,variations);if($pill.hasClass('selected')){$pill.removeClass('disabled')}else{$pill.toggleClass('disabled',!isAvailable)}})});function isOptionAvailable(attrName,optionValue,currentSelections,variations){if(!variations||variations.length===0){return true}const testSelections={...currentSelections};testSelections[attrName]=optionValue;for(let i=0;i<variations.length;i++){const variation=variations[i];if(!variation.is_in_stock||!variation.variation_is_visible){continue}let matches=true;for(const[key,value]of Object.entries(testSelections)){const varAttrKey='attribute_'+key;const varAttrValue=variation.attributes[varAttrKey];if(varAttrValue!==''&&varAttrValue!==value){matches=false;break}}if(matches){return true}}return false}}}$(document).ready(function(){if($('form.variations_form').length){new VariablyPretty()}});$(document.body).on('quick-view-displayed',function(){new VariablyPretty()})})(jQuery);
        </script>
        <?php
    }
    
    public function render_pill_variations($html, $args) {
        global $product;
        
        if (!$product || !$this->is_enabled_for_product($product->get_id())) {
            return $html;
        }
        
        $attribute = $args['attribute'];
        $options = $args['options'];
        $product_id = $args['product']->get_id();
        $selected = $args['selected'];
        
        if (empty($options)) {
            return $html;
        }
        
        // Build pill HTML (WooCommerce already renders the label in templates)
        $pill_html = '<div class="variably-pretty-wrapper" data-attribute="' . esc_attr($attribute) . '">';
        $pill_html .= '<div class="variably-pretty-pills">';
        
        foreach ($options as $option) {
            // Get proper term name/label instead of sanitized slug
            $option_label = $option;
            
            // Check if this is a taxonomy attribute (pa_*)
            if (taxonomy_exists($attribute)) {
                $term = get_term_by('slug', $option, $attribute);
                if ($term && !is_wp_error($term)) {
                    $option_label = $term->name;
                }
            } else {
                // Custom product attribute - apply WC filter
                $option_label = apply_filters('woocommerce_variation_option_name', $option, null, $attribute, $product);
            }
            
            $is_selected = $selected === $option ? 'selected' : '';
            
            $pill_html .= sprintf(
                '<button type="button" class="variably-pretty-pill %s" data-value="%s">%s</button>',
                esc_attr($is_selected),
                esc_attr($option),
                esc_html($option_label)
            );
        }
        
        $pill_html .= '</div>';
        
        // Hidden select for WooCommerce compatibility
        // Keep original attribute name format that WooCommerce expects
        $select_id = sanitize_title($attribute);
        $pill_html .= '<select id="' . esc_attr($select_id) . '" class="variably-pretty-hidden-select" name="attribute_' . esc_attr($attribute) . '" data-attribute_name="attribute_' . esc_attr($attribute) . '" style="display: none;">';
        $pill_html .= '<option value="">Choose an option</option>';
        foreach ($options as $option) {
            // Get proper term name/label
            $option_label = $option;
            if (taxonomy_exists($attribute)) {
                $term = get_term_by('slug', $option, $attribute);
                if ($term && !is_wp_error($term)) {
                    $option_label = $term->name;
                }
            } else {
                $option_label = apply_filters('woocommerce_variation_option_name', $option, null, $attribute, $product);
            }
            
            $pill_html .= sprintf(
                '<option value="%s" %s>%s</option>',
                esc_attr($option),
                selected($selected, $option, false),
                esc_html($option_label)
            );
        }
        $pill_html .= '</select>';
        $pill_html .= '</div>';
        
        return $pill_html;
    }
    
    private function is_enabled_for_product($product_id) {
        $enable_all = get_option('variably_pretty_enable_all', false);
        
        if ($enable_all) {
            return true;
        }
        
        $enabled_products = get_option('variably_pretty_enabled_products', []);
        return in_array($product_id, $enabled_products);
    }
}

// Initialize plugin
Variably_Pretty::instance();
